//
//  Login.app
// 
//  Copyright (c) 1997, 1998 Per Liden
// 
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//


#include "App.h"


extern App* LoginApp;
const char* XServer[] = DEFAULT_XSERVER;

void CatchSignal(int sig)
{
    cerr << APPNAME << ": unexpected signal " << sig << endl;
    LoginApp->StopServer();
    exit(ERR_EXIT);
}


void AlarmSignal(int sig)
{
    int pid = LoginApp->GetServerPID();
    if(waitpid(pid, NULL, WNOHANG) == pid)
    {
	LoginApp->StopServer();
	exit(OK_EXIT);
    }
    signal(sig, AlarmSignal);
    alarm(2);
}


void User1Signal(int sig)
{
    signal(sig, User1Signal);
}


App::App(int argc, char** argv)
{
    int tmp;
    ServerPID = -1;

    // Set default options
    NoAnimations = 0;

    // Parse command line
    while((tmp = getopt(argc, argv, "nvh?")) != EOF) 
    {
	switch (tmp) 
	{
	case 'n':	// No animations
	    NoAnimations = 1;
	    break;	    
	case 'v':	// Version
	    cout << APPNAME << " version " << VERSION << endl;
	    exit(OK_EXIT);
	    break;
	case '?':	// Ilegal
	    cerr << endl;
	case 'h':   // Help
	    cerr << "usage:  " << APPNAME << " [option ...]" << endl
		 << "options:" << endl
		 << "    -noanimations" << endl
		 << "    -version" << endl;
	    exit(OK_EXIT);
	    break;
	}
    }
}


void App::Run()
{
    // Start x-server
    setenv("DISPLAY", DISPLAY, 1);
    signal(SIGQUIT, CatchSignal);
    signal(SIGTERM, CatchSignal);
    signal(SIGKILL, CatchSignal);
    signal(SIGINT, CatchSignal);
    signal(SIGHUP, CatchSignal);
    signal(SIGPIPE, CatchSignal);
    signal(SIGUSR1, User1Signal);
    signal(SIGALRM, AlarmSignal);    
    StartServer((char**)XServer);
    alarm(2);
        
    // Open display
    if((Dpy = XOpenDisplay(DISPLAY)) == 0) 
    {
	cerr << APPNAME << ": could not open display '" << DISPLAY << "'" << endl;
	StopServer();
	exit(ERR_EXIT);
    }

    // Get screen and root window
    Scr = DefaultScreen(Dpy);
    Root = RootWindow(Dpy, Scr);

    // Create panel
    LoginPanel = new Panel(Dpy, Scr, Root);

    // Start looping
    XEvent event;
    int panelclosed = 1;
    int Action;

    while(1) 
    {	
	if(panelclosed)
	{
	    // Init root
	    system(INIT_CMD);

	    // Close all clients
	    KillAllClients(False);
	    KillAllClients(True);
	    
	    // Show panel
	    LoginPanel->OpenPanel();
	}
	
	Action = WAIT;
	LoginPanel->GetInput()->Reset();

	while(Action == WAIT)
	{
	    XNextEvent(Dpy, &event);
	    Action = LoginPanel->EventHandler(&event);	    
	}	

	if(Action == FAIL)
	{
	    panelclosed = 0;
	    if(NoAnimations)
	    {
		LoginPanel->ClearPanel();
		XBell(Dpy, 100);
	    }
	    else
		LoginPanel->ShakePanel();
	}
	else
	{
	    panelclosed = 1;
	    if((Action == LOGIN || Action == CONSOLE) && !NoAnimations)
		LoginPanel->ShrinkPanel();
	    LoginPanel->ClosePanel();
	    
	    switch(Action)
	    {
	    case LOGIN: Login(); break;		
	    case CONSOLE: Console(); break;		
	    case REBOOT: Reboot(); break;
	    case HALT: Halt(); break;
	    case EXIT: Exit(); break;
	    }
	}
    }
}


int App::GetServerPID()
{
    return ServerPID;
}


void App::Login()
{
    struct passwd *pw;
    pid_t pid;

    pw = LoginPanel->GetInput()->GetPasswdStruct();
    if(pw == 0)
	return;
	
    // Create new process
    pid = fork();
    if(pid == 0)
    {
	// Login process starts here
	SwitchUser Su(pw);
        Su.Login(LOGIN_CMD);
	exit(OK_EXIT);
    }

    // Wait until user is logging out (login process terminates)
    pid_t wpid = -1;
    while (wpid != pid)
	wpid = wait(NULL);    

//    waitpid(pid, 0, 0);

    // Close all clients
    KillAllClients(False);
    KillAllClients(True);

    // Send HUP signal to clientgroup
    killpg(pid, SIGHUP);

    // Send TERM signal to clientgroup, if error send KILL
    if(killpg(pid, SIGTERM))
	killpg(pid, SIGKILL);
}


void App::Reboot()
{
    // Stop alarm clock
    alarm(0);

    // Write message
    LoginPanel->Message("System rebooting.");
    sleep(1);
    LoginPanel->Message("System rebooting..");
    sleep(1);
    LoginPanel->Message("System rebooting...");
    sleep(1);
    LoginPanel->Message("System rebooting....");
    sleep(1);

    // Stop server and reboot
    StopServer();
    system(REBOOT_CMD);
    exit(OK_EXIT);
}


void App::Halt()
{
    // Stop alarm clock
    alarm(0);

    // Write message
    LoginPanel->Message("System halting.");
    sleep(1);
    LoginPanel->Message("System halting..");
    sleep(1);
    LoginPanel->Message("System halting...");
    sleep(1);
    LoginPanel->Message("System halting....");
    sleep(1);

    // Stop server and halt
    StopServer();
    system(HALT_CMD);
    exit(OK_EXIT);
}


void App::Console()
{
    int posx = 40;
    int posy = 40;
    int fontx = 9;
    int fonty = 15;
    int width = (XWidthOfScreen(ScreenOfDisplay(Dpy, Scr)) - (posx * 2)) / fontx;
    int height = (XHeightOfScreen(ScreenOfDisplay(Dpy, Scr)) - (posy * 2)) / fonty;

    // Execute console
    char *tmp = new char[strlen(CONSOLE_CMD) + 60];
    sprintf(tmp, CONSOLE_CMD, width, height, posx, posy, fontx, fonty);
    system(tmp);
    delete [] tmp;
}


void App::Exit()
{
    // Deallocate and stop server
    delete LoginPanel;
    StopServer();
    exit(OK_EXIT);
}


int CatchErrors(Display *dpy, XErrorEvent *ev)
{
    return 0;
}


void App::KillAllClients(Bool top)
{
    Window dummywindow;
    Window *children;
    unsigned int nchildren;
    unsigned int i;
    XWindowAttributes attr;

    XSync(Dpy, 0);
    XSetErrorHandler(CatchErrors);

    nchildren = 0;
    XQueryTree(Dpy, Root, &dummywindow, &dummywindow, &children, &nchildren);
    if(!top) 
    {
	for(i=0; i<nchildren; i++)
	{
	    if(XGetWindowAttributes(Dpy, children[i], &attr) && (attr.map_state == IsViewable))
		children[i] = XmuClientWindow(Dpy, children[i]);
	    else
		children[i] = 0;
	}
    }

    for(i=0; i<nchildren; i++) 
    {
	if(children[i])
	    XKillClient(Dpy, children[i]);
    }
    XFree((char *)children);
    
    XSync(Dpy, 0);
    XSetErrorHandler(NULL);    
}


int App::ServerTimeout(int timeout, char* text)
{
    int	i = 0;
    int pidfound = -1;
    static char	*lasttext;
    
    for(;;) 
    {
	pidfound = waitpid(ServerPID, NULL, WNOHANG);
	if(pidfound == ServerPID)
	    break;	
	if(timeout) 
	{
	    if(i == 0 && text != lasttext)
		cerr << endl << APPNAME << ": waiting for " << text;
	    else
		cerr << ".";	    
	}
	if(timeout)
	    sleep(1);
	if(++i > timeout)
	    break;
    }

    if(i > 0) 
	cerr << endl;
    lasttext = text;

    return (ServerPID != pidfound);
}


int App::WaitForServer()
{
    int	ncycles	 = 120;
    int	cycles;
    
    for(cycles = 0; cycles < ncycles; cycles++) 
    {
	if((Dpy = XOpenDisplay(0)))
	{
	    return 1;
	}
	else 
	{
	    if(!ServerTimeout(1, "X server to begin accepting connections")) 
		break;
	}
    }
    
    cerr << "Giving up." << endl;
    
    return 0;
}


int App::StartServer(char** server)
{
    ServerPID = vfork();
    switch(ServerPID) 
    {
    case 0:
	signal(SIGTTIN, SIG_IGN);
	signal(SIGTTOU, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	setpgid(0,getpid());

	// Execute server
	execvp(server[0], server);
	cerr << APPNAME << ": X server could not be started" << endl;
	exit(ERR_EXIT);	
	break;

    case -1:
	break;

    default:
	errno = 0;
	if(!ServerTimeout(0, "")) 
	{
	    ServerPID = -1;
	    break;
	}
	alarm(15);
	pause();
	alarm(0);
	
	// Wait for server to start up
	if(WaitForServer() == 0) 
	{
	    cerr << APPNAME << ": unable to connect to X server" << endl;
	    StopServer();
	    ServerPID = -1;
	    exit(ERR_EXIT);
	}
	break;
    }

    return ServerPID;
}


jmp_buf CloseEnv;
int IgnoreXIO(Display *d)
{
    cerr << APPNAME << ": connection to X server lost." << endl;
    longjmp(CloseEnv, 1);
}


void App::StopServer()
{
    // Stop alars clock and ignore signals
    alarm(0);
    signal(SIGQUIT, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    signal(SIGHUP, SIG_IGN);
    signal(SIGPIPE, SIG_IGN);
    signal(SIGTERM, SIG_DFL);
    signal(SIGKILL, SIG_DFL);
    signal(SIGALRM, SIG_DFL);

    // Catch X error
    XSetIOErrorHandler(IgnoreXIO);
    if(!setjmp(CloseEnv)) 
	XCloseDisplay(Dpy);

    // Send HUP to process group
    errno = 0;
    if((killpg(getpid(), SIGHUP) != 0) && (errno != ESRCH))
	cerr << APPNAME << ": can't send HUP to process group " << getpid() << endl;
    
    // Send TERM to server
    if(ServerPID < 0)
	return;
    errno = 0;
    if(killpg(ServerPID, SIGTERM) < 0) 
    {
	if(errno == EPERM)
	{
	    cerr << APPNAME << ": can't kill X server" << endl;
	    exit(ERR_EXIT);
	}
	if(errno == ESRCH)
	    return;
    }

    // Wait for server to shut down
    if(!ServerTimeout(10, "X server to shut down")) 
    {
	cerr << endl;
	return;
    }
    
    cerr << endl << APPNAME << ":  X server slow to shut down, sending KILL signal." << endl;

    // Send KILL to server
    errno = 0;
    if(killpg(ServerPID, SIGKILL) < 0) 
    {
	if(errno == ESRCH)
	    return;
    }

    // Wait for server to die
    if(ServerTimeout(3, "server to die")) 
    {
	cerr << endl << APPNAME << ": can't kill server" << endl;
	exit(ERR_EXIT);
    }
    cerr << endl;
}

